# BSD 2-Clause License
#
# Apprise - Push Notification Library.
# Copyright (c) 2025, Chris Caron <lead2gold@gmail.com>
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
# 1. Redistributions of source code must retain the above copyright notice,
#    this list of conditions and the following disclaimer.
#
# 2. Redistributions in binary form must reproduce the above copyright notice,
#    this list of conditions and the following disclaimer in the documentation
#    and/or other materials provided with the distribution.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
# ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
# LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
# CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
# SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
# INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
# CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
# ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
# POSSIBILITY OF SUCH DAMAGE.


import re

from .common import SecureMailMode, WebBaseLogin

# To attempt to make this script stupid proof, if we detect an email address
# that is part of the this table, we can pre-use a lot more defaults if they
# aren't otherwise specified on the users input.
EMAIL_TEMPLATES = (
    # Google GMail
    (
        "Google Mail",
        re.compile(
            r"^((?P<label>[^+]+)\+)?(?P<id>[^@]+)@" r"(?P<domain>gmail\.com)$",
            re.I,
        ),
        {
            "port": 587,
            "smtp_host": "smtp.gmail.com",
            "secure": True,
            "secure_mode": SecureMailMode.STARTTLS,
            "login_type": (WebBaseLogin.EMAIL,),
        },
    ),
    # Yandex
    (
        "Yandex",
        re.compile(
            r"^((?P<label>[^+]+)\+)?(?P<id>[^@]+)@"
            r"(?P<domain>yandex\.(com|ru|ua|by|kz|uz|tr|fr))$",
            re.I,
        ),
        {
            "port": 465,
            "smtp_host": "smtp.yandex.ru",
            "secure": True,
            "secure_mode": SecureMailMode.SSL,
            "login_type": (WebBaseLogin.USERID,),
        },
    ),
    # Microsoft Hotmail
    (
        "Microsoft Hotmail",
        re.compile(
            r"^((?P<label>[^+]+)\+)?(?P<id>[^@]+)@"
            r"(?P<domain>(hotmail|live)\.com(\.au)?)$",
            re.I,
        ),
        {
            "port": 587,
            "smtp_host": "smtp-mail.outlook.com",
            "secure": True,
            "secure_mode": SecureMailMode.STARTTLS,
            "login_type": (WebBaseLogin.EMAIL,),
        },
    ),
    # Microsoft Outlook
    (
        "Microsoft Outlook",
        re.compile(
            r"^((?P<label>[^+]+)\+)?(?P<id>[^@]+)@"
            r"(?P<domain>(smtp\.)?outlook\.com(\.au)?)$",
            re.I,
        ),
        {
            "port": 587,
            "smtp_host": "smtp.outlook.com",
            "secure": True,
            "secure_mode": SecureMailMode.STARTTLS,
            "login_type": (WebBaseLogin.EMAIL,),
        },
    ),
    # Microsoft Office 365 (Email Server)
    # You must specify an authenticated sender address in the from= settings
    # and a valid email in the to= to deliver your emails to
    (
        "Microsoft Office 365",
        re.compile(r"^[^@]+@(?P<domain>(smtp\.)?office365\.com)$", re.I),
        {
            "port": 587,
            "smtp_host": "smtp.office365.com",
            "secure": True,
            "secure_mode": SecureMailMode.STARTTLS,
        },
    ),
    # Yahoo Mail
    (
        "Yahoo Mail",
        re.compile(
            r"^((?P<label>[^+]+)\+)?(?P<id>[^@]+)@"
            r"(?P<domain>yahoo\.(ca|com))$",
            re.I,
        ),
        {
            "port": 465,
            "smtp_host": "smtp.mail.yahoo.com",
            "secure": True,
            "secure_mode": SecureMailMode.STARTTLS,
            "login_type": (WebBaseLogin.EMAIL,),
        },
    ),
    # Fast Mail (Series 1)
    (
        "Fast Mail",
        re.compile(
            r"^((?P<label>[^+]+)\+)?(?P<id>[^@]+)@"
            r"(?P<domain>fastmail\.(com|cn|co\.uk|com\.au|de|es|fm|fr|im|"
            r"in|jp|mx|net|nl|org|se|to|tw|uk|us))$",
            re.I,
        ),
        {
            "port": 465,
            "smtp_host": "smtp.fastmail.com",
            "secure": True,
            "secure_mode": SecureMailMode.SSL,
            "login_type": (WebBaseLogin.EMAIL,),
        },
    ),
    # Fast Mail (Series 2)
    (
        "Fast Mail Extended Addresses",
        re.compile(
            r"^((?P<label>[^+]+)\+)?(?P<id>[^@]+)@"
            r"(?P<domain>123mail\.org|airpost\.net|eml\.cc|fmail\.co\.uk|"
            r"fmgirl\.com|fmguy\.com|mailbolt\.com|mailcan\.com|"
            r"mailhaven\.com|mailmight\.com|ml1\.net|mm\.st|myfastmail\.com|"
            r"proinbox\.com|promessage\.com|rushpost\.com|sent\.(as|at|com)|"
            r"speedymail\.org|warpmail\.net|xsmail\.com|150mail\.com|"
            r"150ml\.com|16mail\.com|2-mail\.com|4email\.net|50mail\.com|"
            r"allmail\.net|bestmail\.us|cluemail\.com|elitemail\.org|"
            r"emailcorner\.net|emailengine\.(net|org)|emailgroups\.net|"
            r"emailplus\.org|emailuser\.net|f-m\.fm|fast-email\.com|"
            r"fast-mail\.org|fastem\.com|fastemail\.us|fastemailer\.com|"
            r"fastest\.cc|fastimap\.com|fastmailbox\.net|fastmessaging\.com|"
            r"fea\.st|fmailbox\.com|ftml\.net|h-mail\.us|hailmail\.net|"
            r"imap-mail\.com|imap\.cc|imapmail\.org|inoutbox\.com|"
            r"internet-e-mail\.com|internet-mail\.org|internetemails\.net|"
            r"internetmailing\.net|jetemail\.net|justemail\.net|"
            r"letterboxes\.org|mail-central\.com|mail-page\.com|"
            r"mailandftp\.com|mailas\.com|mailc\.net|mailforce\.net|"
            r"mailftp\.com|mailingaddress\.org|mailite\.com|mailnew\.com|"
            r"mailsent\.net|mailservice\.ms|mailup\.net|mailworks\.org|"
            r"mymacmail\.com|nospammail\.net|ownmail\.net|petml\.com|"
            r"postinbox\.com|postpro\.net|realemail\.net|reallyfast\.biz|"
            r"reallyfast\.info|speedpost\.net|ssl-mail\.com|swift-mail\.com|"
            r"the-fastest\.net|the-quickest\.com|theinternetemail\.com|"
            r"veryfast\.biz|veryspeedy\.net|yepmail\.net)$",
            re.I,
        ),
        {
            "port": 465,
            "smtp_host": "smtp.fastmail.com",
            "secure": True,
            "secure_mode": SecureMailMode.SSL,
            "login_type": (WebBaseLogin.EMAIL,),
        },
    ),
    # Zoho Mail (Free)
    (
        "Zoho Mail",
        re.compile(
            r"^((?P<label>[^+]+)\+)?(?P<id>[^@]+)@"
            r"(?P<domain>zoho(mail)?\.com)$",
            re.I,
        ),
        {
            "port": 587,
            "smtp_host": "smtp.zoho.com",
            "secure": True,
            "secure_mode": SecureMailMode.STARTTLS,
            "login_type": (WebBaseLogin.EMAIL,),
        },
    ),
    # SendGrid (Email Server)
    # You must specify an authenticated sender address in the from= settings
    # and a valid email in the to= to deliver your emails to
    (
        "SendGrid",
        re.compile(
            r"^((?P<label>[^+]+)\+)?(?P<id>[^@]+)@"
            r"(?P<domain>(\.smtp)?sendgrid\.(com|net))$",
            re.I,
        ),
        {
            "port": 465,
            "smtp_host": "smtp.sendgrid.net",
            "secure": True,
            "secure_mode": SecureMailMode.SSL,
            "login_type": (WebBaseLogin.USERID,),
        },
    ),
    # 163.com
    (
        "163.com",
        re.compile(
            r"^((?P<label>[^+]+)\+)?(?P<id>[^@]+)@" r"(?P<domain>163\.com)$",
            re.I,
        ),
        {
            "port": 465,
            "smtp_host": "smtp.163.com",
            "secure": True,
            "secure_mode": SecureMailMode.SSL,
            "login_type": (WebBaseLogin.EMAIL,),
        },
    ),
    # Foxmail.com
    (
        "Foxmail.com",
        re.compile(
            r"^((?P<label>[^+]+)\+)?(?P<id>[^@]+)@"
            r"(?P<domain>(foxmail|qq)\.com)$",
            re.I,
        ),
        {
            "port": 587,
            "smtp_host": "smtp.qq.com",
            "secure": True,
            "secure_mode": SecureMailMode.STARTTLS,
            "login_type": (WebBaseLogin.EMAIL,),
        },
    ),
    # Comcast.net
    (
        "Comcast.net",
        re.compile(
            r"^((?P<label>[^+]+)\+)?(?P<id>[^@]+)@"
            r"(?P<domain>(comcast)\.net)$",
            re.I,
        ),
        {
            "port": 465,
            "smtp_host": "smtp.comcast.net",
            "secure": True,
            "secure_mode": SecureMailMode.SSL,
            "login_type": (WebBaseLogin.EMAIL,),
        },
    ),
    # Localhost handling
    (
        "Local Mail Server",
        re.compile(
            r"^(((?P<label>[^+]+)\+)?(?P<id>[^@]+)@)?"
            r"(?P<domain>localhost(\.localdomain)?)$",
            re.I,
        ),
        {
            # Provide a default user if one isn't provided
            "from_user": "root",
            "smtp_host": None,
        },
    ),
    # Catch All
    (
        "Custom",
        re.compile(
            r"^((?P<label>[^+]+)\+)?(?P<id>[^@]+)@" r"(?P<domain>.+)$", re.I
        ),
        {
            # Setting smtp_host to None is a way of
            # auto-detecting it based on other parameters
            # specified.  There is no reason to ever modify
            # this Catch All
            "smtp_host": None,
        },
    ),
)
