﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/DateTime.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/transcribe/TranscribeService_EXPORTS.h>
#include <aws/transcribe/model/LanguageCode.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Json {
class JsonValue;
class JsonView;
}  // namespace Json
}  // namespace Utils
namespace TranscribeService {
namespace Model {

/**
 * <p>Provides information about a custom vocabulary filter, including the language
 * of the filter, when it was last modified, and its name.</p><p><h3>See Also:</h3>
 * <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/VocabularyFilterInfo">AWS
 * API Reference</a></p>
 */
class VocabularyFilterInfo {
 public:
  AWS_TRANSCRIBESERVICE_API VocabularyFilterInfo() = default;
  AWS_TRANSCRIBESERVICE_API VocabularyFilterInfo(Aws::Utils::Json::JsonView jsonValue);
  AWS_TRANSCRIBESERVICE_API VocabularyFilterInfo& operator=(Aws::Utils::Json::JsonView jsonValue);
  AWS_TRANSCRIBESERVICE_API Aws::Utils::Json::JsonValue Jsonize() const;

  ///@{
  /**
   * <p>A unique name, chosen by you, for your custom vocabulary filter. This name is
   * case sensitive, cannot contain spaces, and must be unique within an Amazon Web
   * Services account.</p>
   */
  inline const Aws::String& GetVocabularyFilterName() const { return m_vocabularyFilterName; }
  inline bool VocabularyFilterNameHasBeenSet() const { return m_vocabularyFilterNameHasBeenSet; }
  template <typename VocabularyFilterNameT = Aws::String>
  void SetVocabularyFilterName(VocabularyFilterNameT&& value) {
    m_vocabularyFilterNameHasBeenSet = true;
    m_vocabularyFilterName = std::forward<VocabularyFilterNameT>(value);
  }
  template <typename VocabularyFilterNameT = Aws::String>
  VocabularyFilterInfo& WithVocabularyFilterName(VocabularyFilterNameT&& value) {
    SetVocabularyFilterName(std::forward<VocabularyFilterNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The language code that represents the language of the entries in your
   * vocabulary filter. Each custom vocabulary filter must contain terms in only one
   * language.</p> <p>A custom vocabulary filter can only be used to transcribe files
   * in the same language as the filter. For example, if you create a custom
   * vocabulary filter using US English (<code>en-US</code>), you can only apply this
   * filter to files that contain English audio.</p> <p>For a list of supported
   * languages and their associated language codes, refer to the <a
   * href="https://docs.aws.amazon.com/transcribe/latest/dg/supported-languages.html">Supported
   * languages</a> table.</p>
   */
  inline LanguageCode GetLanguageCode() const { return m_languageCode; }
  inline bool LanguageCodeHasBeenSet() const { return m_languageCodeHasBeenSet; }
  inline void SetLanguageCode(LanguageCode value) {
    m_languageCodeHasBeenSet = true;
    m_languageCode = value;
  }
  inline VocabularyFilterInfo& WithLanguageCode(LanguageCode value) {
    SetLanguageCode(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The date and time the specified custom vocabulary filter was last
   * modified.</p> <p>Timestamps are in the format
   * <code>YYYY-MM-DD'T'HH:MM:SS.SSSSSS-UTC</code>. For example,
   * <code>2022-05-04T12:32:58.761000-07:00</code> represents 12:32 PM UTC-7 on May
   * 4, 2022.</p>
   */
  inline const Aws::Utils::DateTime& GetLastModifiedTime() const { return m_lastModifiedTime; }
  inline bool LastModifiedTimeHasBeenSet() const { return m_lastModifiedTimeHasBeenSet; }
  template <typename LastModifiedTimeT = Aws::Utils::DateTime>
  void SetLastModifiedTime(LastModifiedTimeT&& value) {
    m_lastModifiedTimeHasBeenSet = true;
    m_lastModifiedTime = std::forward<LastModifiedTimeT>(value);
  }
  template <typename LastModifiedTimeT = Aws::Utils::DateTime>
  VocabularyFilterInfo& WithLastModifiedTime(LastModifiedTimeT&& value) {
    SetLastModifiedTime(std::forward<LastModifiedTimeT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_vocabularyFilterName;

  LanguageCode m_languageCode{LanguageCode::NOT_SET};

  Aws::Utils::DateTime m_lastModifiedTime{};
  bool m_vocabularyFilterNameHasBeenSet = false;
  bool m_languageCodeHasBeenSet = false;
  bool m_lastModifiedTimeHasBeenSet = false;
};

}  // namespace Model
}  // namespace TranscribeService
}  // namespace Aws
