﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/kinesisanalytics/KinesisAnalytics_EXPORTS.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Json {
class JsonValue;
class JsonView;
}  // namespace Json
}  // namespace Utils
namespace KinesisAnalytics {
namespace Model {

/**
 * <p>When configuring application output, identifies an AWS Lambda function as the
 * destination. You provide the function Amazon Resource Name (ARN) and also an IAM
 * role ARN that Amazon Kinesis Analytics can use to write to the function on your
 * behalf. </p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/kinesisanalytics-2015-08-14/LambdaOutput">AWS
 * API Reference</a></p>
 */
class LambdaOutput {
 public:
  AWS_KINESISANALYTICS_API LambdaOutput() = default;
  AWS_KINESISANALYTICS_API LambdaOutput(Aws::Utils::Json::JsonView jsonValue);
  AWS_KINESISANALYTICS_API LambdaOutput& operator=(Aws::Utils::Json::JsonView jsonValue);
  AWS_KINESISANALYTICS_API Aws::Utils::Json::JsonValue Jsonize() const;

  ///@{
  /**
   * <p>Amazon Resource Name (ARN) of the destination Lambda function to write
   * to.</p>  <p>To specify an earlier version of the Lambda function than the
   * latest, include the Lambda function version in the Lambda function ARN. For more
   * information about Lambda ARNs, see <a
   * href="/general/latest/gr/aws-arns-and-namespaces.html#arn-syntax-lambda">Example
   * ARNs: AWS Lambda</a> </p>
   */
  inline const Aws::String& GetResourceARN() const { return m_resourceARN; }
  inline bool ResourceARNHasBeenSet() const { return m_resourceARNHasBeenSet; }
  template <typename ResourceARNT = Aws::String>
  void SetResourceARN(ResourceARNT&& value) {
    m_resourceARNHasBeenSet = true;
    m_resourceARN = std::forward<ResourceARNT>(value);
  }
  template <typename ResourceARNT = Aws::String>
  LambdaOutput& WithResourceARN(ResourceARNT&& value) {
    SetResourceARN(std::forward<ResourceARNT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>ARN of the IAM role that Amazon Kinesis Analytics can assume to write to the
   * destination function on your behalf. You need to grant the necessary permissions
   * to this role. </p>
   */
  inline const Aws::String& GetRoleARN() const { return m_roleARN; }
  inline bool RoleARNHasBeenSet() const { return m_roleARNHasBeenSet; }
  template <typename RoleARNT = Aws::String>
  void SetRoleARN(RoleARNT&& value) {
    m_roleARNHasBeenSet = true;
    m_roleARN = std::forward<RoleARNT>(value);
  }
  template <typename RoleARNT = Aws::String>
  LambdaOutput& WithRoleARN(RoleARNT&& value) {
    SetRoleARN(std::forward<RoleARNT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_resourceARN;

  Aws::String m_roleARN;
  bool m_resourceARNHasBeenSet = false;
  bool m_roleARNHasBeenSet = false;
};

}  // namespace Model
}  // namespace KinesisAnalytics
}  // namespace Aws
