﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/DateTime.h>
#include <aws/core/utils/memory/stl/AWSMap.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/managedblockchain/ManagedBlockchain_EXPORTS.h>
#include <aws/managedblockchain/model/AccessorNetworkType.h>
#include <aws/managedblockchain/model/AccessorStatus.h>
#include <aws/managedblockchain/model/AccessorType.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Json {
class JsonValue;
class JsonView;
}  // namespace Json
}  // namespace Utils
namespace ManagedBlockchain {
namespace Model {

/**
 * <p>The properties of the Accessor.</p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/managedblockchain-2018-09-24/Accessor">AWS
 * API Reference</a></p>
 */
class Accessor {
 public:
  AWS_MANAGEDBLOCKCHAIN_API Accessor() = default;
  AWS_MANAGEDBLOCKCHAIN_API Accessor(Aws::Utils::Json::JsonView jsonValue);
  AWS_MANAGEDBLOCKCHAIN_API Accessor& operator=(Aws::Utils::Json::JsonView jsonValue);
  AWS_MANAGEDBLOCKCHAIN_API Aws::Utils::Json::JsonValue Jsonize() const;

  ///@{
  /**
   * <p>The unique identifier of the accessor.</p>
   */
  inline const Aws::String& GetId() const { return m_id; }
  inline bool IdHasBeenSet() const { return m_idHasBeenSet; }
  template <typename IdT = Aws::String>
  void SetId(IdT&& value) {
    m_idHasBeenSet = true;
    m_id = std::forward<IdT>(value);
  }
  template <typename IdT = Aws::String>
  Accessor& WithId(IdT&& value) {
    SetId(std::forward<IdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The type of the accessor.</p>  <p>Currently, accessor type is
   * restricted to <code>BILLING_TOKEN</code>.</p>
   */
  inline AccessorType GetType() const { return m_type; }
  inline bool TypeHasBeenSet() const { return m_typeHasBeenSet; }
  inline void SetType(AccessorType value) {
    m_typeHasBeenSet = true;
    m_type = value;
  }
  inline Accessor& WithType(AccessorType value) {
    SetType(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The billing token is a property of the Accessor. Use this token to when
   * making calls to the blockchain network. The billing token is used to track your
   * accessor token for billing requests.</p>
   */
  inline const Aws::String& GetBillingToken() const { return m_billingToken; }
  inline bool BillingTokenHasBeenSet() const { return m_billingTokenHasBeenSet; }
  template <typename BillingTokenT = Aws::String>
  void SetBillingToken(BillingTokenT&& value) {
    m_billingTokenHasBeenSet = true;
    m_billingToken = std::forward<BillingTokenT>(value);
  }
  template <typename BillingTokenT = Aws::String>
  Accessor& WithBillingToken(BillingTokenT&& value) {
    SetBillingToken(std::forward<BillingTokenT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The current status of the accessor.</p>
   */
  inline AccessorStatus GetStatus() const { return m_status; }
  inline bool StatusHasBeenSet() const { return m_statusHasBeenSet; }
  inline void SetStatus(AccessorStatus value) {
    m_statusHasBeenSet = true;
    m_status = value;
  }
  inline Accessor& WithStatus(AccessorStatus value) {
    SetStatus(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The creation date and time of the accessor.</p>
   */
  inline const Aws::Utils::DateTime& GetCreationDate() const { return m_creationDate; }
  inline bool CreationDateHasBeenSet() const { return m_creationDateHasBeenSet; }
  template <typename CreationDateT = Aws::Utils::DateTime>
  void SetCreationDate(CreationDateT&& value) {
    m_creationDateHasBeenSet = true;
    m_creationDate = std::forward<CreationDateT>(value);
  }
  template <typename CreationDateT = Aws::Utils::DateTime>
  Accessor& WithCreationDate(CreationDateT&& value) {
    SetCreationDate(std::forward<CreationDateT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of the accessor. For more information about
   * ARNs and their format, see <a
   * href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">Amazon
   * Resource Names (ARNs)</a> in the <i>Amazon Web Services General
   * Reference</i>.</p>
   */
  inline const Aws::String& GetArn() const { return m_arn; }
  inline bool ArnHasBeenSet() const { return m_arnHasBeenSet; }
  template <typename ArnT = Aws::String>
  void SetArn(ArnT&& value) {
    m_arnHasBeenSet = true;
    m_arn = std::forward<ArnT>(value);
  }
  template <typename ArnT = Aws::String>
  Accessor& WithArn(ArnT&& value) {
    SetArn(std::forward<ArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The tags assigned to the Accessor.</p> <p>For more information about tags,
   * see <a
   * href="https://docs.aws.amazon.com/managed-blockchain/latest/ethereum-dev/tagging-resources.html">Tagging
   * Resources</a> in the <i>Amazon Managed Blockchain Ethereum Developer Guide</i>,
   * or <a
   * href="https://docs.aws.amazon.com/managed-blockchain/latest/hyperledger-fabric-dev/tagging-resources.html">Tagging
   * Resources</a> in the <i>Amazon Managed Blockchain Hyperledger Fabric Developer
   * Guide</i>.</p>
   */
  inline const Aws::Map<Aws::String, Aws::String>& GetTags() const { return m_tags; }
  inline bool TagsHasBeenSet() const { return m_tagsHasBeenSet; }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  Accessor& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsKeyT = Aws::String, typename TagsValueT = Aws::String>
  Accessor& AddTags(TagsKeyT&& key, TagsValueT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace(std::forward<TagsKeyT>(key), std::forward<TagsValueT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The blockchain network that the Accessor token is created for.</p>
   */
  inline AccessorNetworkType GetNetworkType() const { return m_networkType; }
  inline bool NetworkTypeHasBeenSet() const { return m_networkTypeHasBeenSet; }
  inline void SetNetworkType(AccessorNetworkType value) {
    m_networkTypeHasBeenSet = true;
    m_networkType = value;
  }
  inline Accessor& WithNetworkType(AccessorNetworkType value) {
    SetNetworkType(value);
    return *this;
  }
  ///@}
 private:
  Aws::String m_id;

  AccessorType m_type{AccessorType::NOT_SET};

  Aws::String m_billingToken;

  AccessorStatus m_status{AccessorStatus::NOT_SET};

  Aws::Utils::DateTime m_creationDate{};

  Aws::String m_arn;

  Aws::Map<Aws::String, Aws::String> m_tags;

  AccessorNetworkType m_networkType{AccessorNetworkType::NOT_SET};
  bool m_idHasBeenSet = false;
  bool m_typeHasBeenSet = false;
  bool m_billingTokenHasBeenSet = false;
  bool m_statusHasBeenSet = false;
  bool m_creationDateHasBeenSet = false;
  bool m_arnHasBeenSet = false;
  bool m_tagsHasBeenSet = false;
  bool m_networkTypeHasBeenSet = false;
};

}  // namespace Model
}  // namespace ManagedBlockchain
}  // namespace Aws
